"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PropertyOrVariable = exports.FunctionOrMethodWithoutBody = exports.FunctionOrMethodWithBody = exports.FunctionOrMethod = exports.Enum = exports.Interface = exports.Class = exports.TypeAlias = exports.SourceFileElement = exports.Namespace = exports.BaseModelElementWithNode = exports.BaseModelElement = exports.Root = exports.Dependency = exports.DependencyKind = exports.ElementType = void 0;
exports.getElementById = getElementById;
exports.lookupSymbol = lookupSymbol;
exports.lookupNode = lookupNode;
exports.getAllElements = getAllElements;
exports.getSourceFileOf = getSourceFileOf;
const metric_collector_1 = require("./metric_collector");
var ElementType;
(function (ElementType) {
    ElementType[ElementType["Root"] = 0] = "Root";
    ElementType[ElementType["SourceFile"] = 1] = "SourceFile";
    ElementType[ElementType["Function"] = 2] = "Function";
    ElementType[ElementType["Constructor"] = 3] = "Constructor";
    ElementType[ElementType["Method"] = 4] = "Method";
    ElementType[ElementType["Variable"] = 5] = "Variable";
    ElementType[ElementType["Property"] = 6] = "Property";
    ElementType[ElementType["Getter"] = 7] = "Getter";
    ElementType[ElementType["Setter"] = 8] = "Setter";
    ElementType[ElementType["Class"] = 9] = "Class";
    ElementType[ElementType["Interface"] = 10] = "Interface";
    ElementType[ElementType["Enum"] = 11] = "Enum";
    ElementType[ElementType["EnumMember"] = 12] = "EnumMember";
    ElementType[ElementType["Namespace"] = 13] = "Namespace";
    ElementType[ElementType["TypeAlias"] = 14] = "TypeAlias";
})(ElementType || (exports.ElementType = ElementType = {}));
var DependencyKind;
(function (DependencyKind) {
    DependencyKind[DependencyKind["READ"] = 0] = "READ";
    DependencyKind[DependencyKind["WRITE"] = 1] = "WRITE";
    DependencyKind[DependencyKind["CALL"] = 2] = "CALL";
    DependencyKind[DependencyKind["NEW"] = 3] = "NEW";
    DependencyKind[DependencyKind["EXTENDS"] = 4] = "EXTENDS";
    DependencyKind[DependencyKind["IMPLEMENTS"] = 5] = "IMPLEMENTS";
    DependencyKind[DependencyKind["USES"] = 6] = "USES";
    DependencyKind[DependencyKind["DECORATES"] = 7] = "DECORATES";
    DependencyKind[DependencyKind["IMPORTS"] = 8] = "IMPORTS";
})(DependencyKind || (exports.DependencyKind = DependencyKind = {}));
let nextId = 2;
const elements = [];
const symbolTable = new Map();
const alternateSymbolTable = new Map();
class Dependency {
    constructor(fromId, toId, line, kind) {
        this.fromId = fromId;
        this.toId = toId;
        this.line = line;
        this.kind = kind;
    }
    toJson() {
        return [this.fromId, this.toId, this.line, DependencyKind[this.kind]];
    }
}
exports.Dependency = Dependency;
class Root {
    constructor(tsConfigPath) {
        this.children = [];
        this.tsConfigPath = tsConfigPath;
        elements.push(this);
        this.dependencies = [];
    }
    get name() {
        return "Root";
    }
    get fullName() {
        return this.name;
    }
    get type() {
        return ElementType.Root;
    }
    get parent() {
        return null;
    }
    get node() {
        return null;
    }
    get line() {
        return undefined;
    }
    get id() {
        return 1;
    }
    get sourceFile() {
        return undefined;
    }
    get isExternal() {
        return undefined;
    }
    get isNameContributor() {
        return false;
    }
    get allChildren() {
        const result = [];
        this.children.forEach(child => child.allChildren.forEach(c => result.push(c)));
        return result;
    }
    addDependency(from, to, line, kind) {
        this.dependencies.push(new Dependency(from.id, to.id, line, kind));
    }
    toJson() {
        return { tsConfigPath: this.tsConfigPath, "numberOfElements": nextId - 2, children: this.children.map((val, index) => val.toJson()), dependencies: this.dependencies.map((val, index) => val.toJson()) };
    }
}
exports.Root = Root;
class BaseModelElement {
    constructor(parent, name) {
        this.parent = parent;
        this.name = name;
        this.children = [];
        parent.children.push(this);
        this.id = nextId++;
        elements.push(this);
    }
    get sourceFile() {
        return this.parent.sourceFile;
    }
    get isExternal() {
        return this.parent.isExternal;
    }
    get isNameContributor() {
        return false;
    }
    get fullName() {
        if (this.parent.isNameContributor) {
            return this.parent.fullName + "." + this.name;
        }
        return this.name;
    }
    get allChildren() {
        const result = [];
        this.children.forEach(child => child.allChildren.forEach(c => result.push(c)));
        return result;
    }
    toJson() {
        return { idNameKind: [this.id, this.name, ElementType[this.type]], children: this.children.map((value, index) => value.toJson()) };
    }
}
exports.BaseModelElement = BaseModelElement;
class BaseModelElementWithNode extends BaseModelElement {
    constructor(parent, name, node, line) {
        super(parent, name);
        this.name = name;
        this.node = node;
        this.line = line;
        const sym = node.getSymbol();
        const compilerNode = node.compilerNode;
        if (sym) {
            let definitions = symbolTable.get(sym);
            if (!definitions) {
                definitions = [];
                symbolTable.set(sym, definitions);
            }
            definitions.push(this);
        }
        alternateSymbolTable.set(compilerNode, this);
    }
    toJson() {
        let result = super.toJson();
        result.line = this.line;
        return result;
    }
}
exports.BaseModelElementWithNode = BaseModelElementWithNode;
class Namespace extends BaseModelElementWithNode {
    constructor(parent, name, node, line) {
        super(parent, name, node, line);
    }
    get isNameContributor() {
        return true;
    }
    get type() {
        return ElementType.Namespace;
    }
}
exports.Namespace = Namespace;
class SourceFileElement extends BaseModelElement {
    constructor(parent, name, node, isExternal = false) {
        super(parent, name);
        this.node = node;
        this.numberOfStatements = -1;
        this.cyclomaticComplexity = -1;
        this.modifiedCyclomaticComplexity = -1;
        this.logicalOperations = -1;
        this.maxIndent = -1;
        this.hasBeenParsed = false;
        this.elementMap = new Map();
        this.importMap = new Map();
        this._isExternal = isExternal;
    }
    get type() {
        return ElementType.SourceFile;
    }
    get line() {
        return undefined;
    }
    get sourceFile() {
        return this.name;
    }
    get isExternal() {
        return this._isExternal;
    }
    addElement(element) {
        const fullName = element.fullName;
        let elements = this.elementMap.get(fullName);
        if (!elements) {
            elements = [];
            this.elementMap.set(fullName, elements);
        }
        elements.push(element);
    }
    findElements(name) {
        const result = this.elementMap.get(name);
        if (!result) {
            return this.getImportsByName(name);
        }
        return result;
    }
    defineImport(name, elements) {
        this.importMap.set(name, elements);
    }
    getImportsByName(name) {
        let result = this.importMap.get(name);
        if (!result) {
            return undefined;
        }
        return result;
    }
    toJson() {
        let result = super.toJson();
        result.isExternal = this._isExternal;
        if (this.numberOfStatements >= 0)
            result.bodyMetrics = [this.numberOfStatements, this.cyclomaticComplexity, this.modifiedCyclomaticComplexity, this.logicalOperations, this.maxIndent];
        return result;
    }
}
exports.SourceFileElement = SourceFileElement;
class TypeAlias extends BaseModelElementWithNode {
    constructor(parent, name, node, line) {
        super(parent, name, node, line);
        this.typeParameters = "";
    }
    get type() {
        return ElementType.TypeAlias;
    }
    toJson() {
        let result = super.toJson();
        result.typeParameters = this.typeParameters;
        return result;
    }
}
exports.TypeAlias = TypeAlias;
class Class extends BaseModelElementWithNode {
    constructor(parent, name, node, line, isAbstract) {
        super(parent, name, node, line);
        this.isAbstract = isAbstract;
        this.typeParameters = "";
    }
    get type() {
        return ElementType.Class;
    }
    get isNameContributor() {
        return true;
    }
    toJson() {
        let result = super.toJson();
        result.isAbstract = this.isAbstract;
        result.typeParameters = this.typeParameters;
        return result;
    }
}
exports.Class = Class;
class Interface extends BaseModelElementWithNode {
    constructor(parent, name, node, line) {
        super(parent, name, node, line);
        this.typeParameters = "";
    }
    get type() {
        return ElementType.Interface;
    }
    get isNameContributor() {
        return true;
    }
    toJson() {
        let result = super.toJson();
        result.typeParameters = this.typeParameters;
        return result;
    }
}
exports.Interface = Interface;
class Enum extends BaseModelElementWithNode {
    constructor(parent, name, node, line) {
        super(parent, name, node, line);
    }
    get isNameContributor() {
        return true;
    }
    get type() {
        return ElementType.Enum;
    }
}
exports.Enum = Enum;
class FunctionOrMethod extends BaseModelElementWithNode {
    constructor(parent, name, node, line) {
        super(parent, name, node, line);
        this.numberOfParameters = 0;
        this.parameters = "";
        this.typeParameters = "";
        this.returnType = "";
    }
    toJson() {
        let result = super.toJson();
        result.numberOfParameters = this.numberOfParameters;
        result.isAbstract = this.isAbstract;
        result.parameters = this.parameters;
        result.typeParameters = this.typeParameters;
        result.returnType = this.returnType;
        return result;
    }
}
exports.FunctionOrMethod = FunctionOrMethod;
class FunctionOrMethodWithBody extends FunctionOrMethod {
    constructor(parent, name, node, line, type, bodyNode) {
        super(parent, name, node, line);
        this.type = type;
        this.bodyNode = bodyNode;
        this.numberOfStatements = -1;
        this.cyclomaticComplexity = -1;
        this.modifiedCyclomaticComplexity = -1;
        this.logicalOperations = -1;
        this.maxIndent = -1;
        const mc = new metric_collector_1.MetricCollector(this.bodyNode);
        mc.collectMetrics();
        this.numberOfStatements = mc.numberOfStatements;
        this.cyclomaticComplexity = mc.cyclomaticComplexity;
        this.modifiedCyclomaticComplexity = mc.modifiedCyclomaticComplexity;
        this.logicalOperations = mc.logicalOperations;
        this.maxIndent = mc.maxIndent;
    }
    get isAbstract() {
        return false;
    }
    toJson() {
        let result = super.toJson();
        result.hasBody = true;
        result.bodyMetrics = [this.numberOfStatements, this.cyclomaticComplexity, this.modifiedCyclomaticComplexity, this.logicalOperations, this.maxIndent];
        return result;
    }
}
exports.FunctionOrMethodWithBody = FunctionOrMethodWithBody;
class FunctionOrMethodWithoutBody extends FunctionOrMethod {
    // type can be Method, Getter or Setter
    constructor(parent, name, node, line, type, isAbstract) {
        super(parent, name, node, line);
        this.type = type;
        this.isAbstract = isAbstract;
    }
    toJson() {
        let result = super.toJson();
        result.hasBody = false;
        return result;
    }
}
exports.FunctionOrMethodWithoutBody = FunctionOrMethodWithoutBody;
class PropertyOrVariable extends BaseModelElementWithNode {
    constructor(parent, name, node, line, type, initializer) {
        super(parent, name, node, line);
        this.type = type;
        this.initializer = initializer;
    }
}
exports.PropertyOrVariable = PropertyOrVariable;
function getElementById(id) {
    return elements[id - 1];
}
function lookupSymbol(symbol) {
    const elements = symbolTable.get(symbol);
    if (!elements)
        return undefined;
    for (let element of elements) {
        if (element instanceof Namespace)
            continue;
        return element;
    }
    return elements[0];
}
function lookupNode(node) {
    return alternateSymbolTable.get(node.compilerNode);
}
function getAllElements() {
    return elements;
}
function getSourceFileOf(element) {
    if (element instanceof SourceFileElement)
        return element;
    if (!element.parent)
        return undefined;
    return getSourceFileOf(element.parent);
}
//# sourceMappingURL=model.js.map