"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.visitSourceFiles = visitSourceFiles;
const model_1 = require("./model");
const ts_morph_1 = require("ts-morph");
const metric_collector_1 = require("./metric_collector");
// We push 'null' on the stack to signal children of local variables not relevant for Sonargraph
const parentStack = [];
const localSymbols = new Set();
const emptySet = new Set();
const externalSourcesToParse = new Set();
let root;
let delayedActions = [];
let currentSource;
function getCurrentParent() {
    return parentStack[parentStack.length - 1];
}
function getLastNonNullParent() {
    for (let i = parentStack.length - 1; i >= 0; i--) {
        const parent = parentStack[i];
        if (parent)
            return parent;
    }
    throw new Error("Parent stack only contains null pointers");
}
function handleParameterTypes(fd, parameters, recordTypeDependencies = true, returnType) {
    parameters.forEach(param => {
        const sym = param.getSymbol();
        if (sym) {
            localSymbols.add(sym);
        }
        if (param.getDecorators().length > 0) {
            handleDecorators(fd, param.getDecorators());
        }
        if (recordTypeDependencies) {
            addDependencyToType(fd, param.getType(), new Set(), param.getStartLineNumber());
        }
    });
    if (returnType && recordTypeDependencies) {
        addDependencyToType(fd, returnType.getType(), new Set(), returnType.getStartLineNumber());
    }
}
function handleDecorators(from, decorators) {
    decorators.forEach(decorator => {
        var _a;
        let decoratorName = decorator.getText().substring(1);
        const pos = decoratorName.indexOf("(");
        if (pos > 0)
            decoratorName = decoratorName.substring(0, pos);
        const elements = (_a = (0, model_1.getSourceFileOf)(from)) === null || _a === void 0 ? void 0 : _a.findElements(decoratorName);
        let target = null;
        if (elements) {
            for (let element of elements) {
                if (element instanceof model_1.FunctionOrMethod) {
                    target = element;
                    break;
                }
            }
            if (target)
                addDependency(from, target, model_1.DependencyKind.DECORATES, decorator.getStartLineNumber());
        }
        if (!target)
            addDependencyToNode(from, decorator, model_1.DependencyKind.DECORATES);
        decorator.getArguments().forEach(arg => {
            detectDependencies(arg, emptySet, from, model_1.DependencyKind.USES);
        });
    });
}
function handlePropertyOrVariable(v, typeNode) {
    addDependencyToType(v, typeNode.getType(), new Set(), typeNode.getStartLineNumber());
}
function handleParametersAndReturnType(fd, params, typeParams, returnTypeNode) {
    let paramsAsString = [];
    let typeParamsAsString = [];
    let returnType = "void";
    params.forEach(pd => paramsAsString.push(pd.getText()));
    typeParams.forEach(tp => typeParamsAsString.push(tp.getText()));
    if (returnTypeNode)
        returnType = returnTypeNode.getText();
    fd.numberOfParameters = params.length;
    fd.parameters = paramsAsString.join(";");
    fd.typeParameters = typeParamsAsString.join(";");
    fd.returnType = returnType;
}
class FunctionDeclarationHandler {
    enter(node) {
        let sym = node.getSymbol();
        let fd;
        let parent = getCurrentParent();
        if (!parent || !sym)
            fd = null;
        else if (node.getBody())
            fd = new model_1.FunctionOrMethodWithBody(parent, sym.getName(), node, node.getStartLineNumber(), model_1.ElementType.Function, node.getBody());
        else
            fd = new model_1.FunctionOrMethodWithoutBody(parent, sym.getName(), node, node.getStartLineNumber(), model_1.ElementType.Function, false);
        if (fd) {
            currentSource.addElement(fd);
            handleParametersAndReturnType(fd, node.getParameters(), node.getTypeParameters(), node.getReturnTypeNode());
            if (!currentSource.isExternal) {
                delayedActions.push(() => handleParameterTypes(fd, node.getParameters(), true, node.getReturnTypeNode()));
            }
        }
        parentStack.push(fd);
    }
}
class ConstructorDeclarationHandler {
    enter(node) {
        let sym = node.getSymbol();
        let parent = getCurrentParent();
        if (parent && sym) {
            let fd;
            if (node.getBody())
                fd = new model_1.FunctionOrMethodWithBody(parent, sym.getName(), node, node.getStartLineNumber(), model_1.ElementType.Constructor, node.getBody());
            else
                fd = new model_1.FunctionOrMethodWithoutBody(parent, sym.getName(), node, node.getStartLineNumber(), model_1.ElementType.Constructor, false);
            currentSource.addElement(fd);
            if (!currentSource.isExternal)
                delayedActions.push(() => handleParameterTypes(fd, node.getParameters()));
            handleParametersAndReturnType(fd, node.getParameters(), node.getTypeParameters(), undefined);
            parentStack.push(fd);
        }
        else
            parentStack.push(null);
    }
}
class MethodDeclarationHandler {
    enter(node) {
        let sym = node.getSymbol();
        let fd;
        let parent = getCurrentParent();
        if (parent && sym) {
            if (node.isAbstract() || !node.getBody())
                fd = new model_1.FunctionOrMethodWithoutBody(parent, sym.getName(), node, node.getStartLineNumber(), model_1.ElementType.Method, node.isAbstract());
            else
                fd = new model_1.FunctionOrMethodWithBody(parent, sym.getName(), node, node.getStartLineNumber(), model_1.ElementType.Method, node.getBody());
            handleParametersAndReturnType(fd, node.getParameters(), node.getTypeParameters(), node.getReturnTypeNode());
            currentSource.addElement(fd);
            if (!currentSource.isExternal) {
                if (node.getDecorators().length > 0) {
                    delayedActions.push(() => handleDecorators(fd, node.getDecorators()));
                }
                delayedActions.push(() => handleParameterTypes(fd, node.getParameters(), true, node.getReturnTypeNode()));
            }
            parentStack.push(fd);
        }
        else
            parentStack.push(null);
    }
}
class MethodSignatureHandler {
    enter(node) {
        let sym = node.getSymbol();
        let parent = getCurrentParent();
        if (parent && sym) {
            let fd = new model_1.FunctionOrMethodWithoutBody(parent, sym.getName(), node, node.getStartLineNumber(), model_1.ElementType.Method, false);
            handleParametersAndReturnType(fd, node.getParameters(), node.getTypeParameters(), node.getReturnTypeNode());
            if (!fd.isExternal)
                delayedActions.push(() => handleParameterTypes(fd, node.getParameters(), true, node.getReturnTypeNode()));
            parentStack.push(fd);
        }
        else
            parentStack.push(null);
    }
}
function handleInheritance(from, clauses) {
    for (const clause of clauses) {
        const isExtends = clause.getToken() == ts_morph_1.SyntaxKind.ExtendsKeyword;
        for (const tn of clause.getTypeNodes()) {
            addDependencyToType(from, tn.getType(), new Set(), tn.getStartLineNumber(), isExtends ? model_1.DependencyKind.EXTENDS : model_1.DependencyKind.IMPLEMENTS);
        }
    }
}
class ClassDeclarationHandler {
    enter(node) {
        let sym = node.getSymbol();
        let parent = getCurrentParent();
        if (parent && sym) {
            let cls = new model_1.Class(parent, sym.getName(), node, node.getStartLineNumber(), node.isAbstract());
            let typeParamsAsString = [];
            currentSource.addElement(cls);
            for (const tp of node.getTypeParameters()) {
                const sym = tp.getSymbol();
                if (sym)
                    localSymbols.add(sym);
                typeParamsAsString.push(tp.getText());
            }
            cls.typeParameters = typeParamsAsString.join(";");
            if (!currentSource.isExternal) {
                delayedActions.push(() => handleInheritance(cls, node.getHeritageClauses()));
                if (node.getDecorators().length > 0) {
                    delayedActions.push(() => handleDecorators(cls, node.getDecorators()));
                }
            }
            parentStack.push(cls);
        }
        else
            parentStack.push(null);
    }
}
class InterfaceDeclarationHandler {
    enter(node) {
        let sym = node.getSymbol();
        let parent = getCurrentParent();
        if (parent && sym) {
            let i = new model_1.Interface(parent, sym.getName(), node, node.getStartLineNumber());
            let typeParamsAsString = [];
            currentSource.addElement(i);
            for (const tp of node.getTypeParameters()) {
                const sym = tp.getSymbol();
                if (sym)
                    localSymbols.add(sym);
                typeParamsAsString.push(tp.getText());
            }
            i.typeParameters = typeParamsAsString.join(";");
            if (!currentSource.isExternal)
                delayedActions.push(() => handleInheritance(i, node.getHeritageClauses()));
            parentStack.push(i);
        }
        else
            parentStack.push(null);
    }
}
class EnumDeclarationHandler {
    enter(node) {
        let sym = node.getSymbol();
        let parent = getCurrentParent();
        if (parent && sym) {
            let e = new model_1.Enum(parent, sym.getName(), node, node.getStartLineNumber());
            currentSource.addElement(e);
            parentStack.push(e);
        }
        else
            parentStack.push(null);
    }
}
class PropertyDeclarationHandler {
    enter(node) {
        let sym = node.getSymbol();
        if (sym) {
            const parent = getCurrentParent();
            if (parent) {
                const prop = new model_1.PropertyOrVariable(parent, sym.getName(), node, node.getStartLineNumber(), model_1.ElementType.Property, node.getInitializer());
                currentSource.addElement(prop);
                parentStack.push(prop);
                if (!currentSource.isExternal) {
                    if (node.getTypeNode()) {
                        delayedActions.push(() => handlePropertyOrVariable(prop, node.getTypeNode()));
                    }
                    if (node.getDecorators().length > 0) {
                        delayedActions.push(() => handleDecorators(prop, node.getDecorators()));
                    }
                }
            }
            else
                parentStack.push(null);
        }
        else
            parentStack.push(null);
    }
}
class PropertySignatureHandler {
    enter(node) {
        let sym = node.getSymbol();
        if (sym) {
            const parent = getCurrentParent();
            if (parent) {
                let prop = new model_1.PropertyOrVariable(parent, sym.getName(), node, node.getStartLineNumber(), model_1.ElementType.Property, node.getInitializer());
                currentSource.addElement(prop);
                parentStack.push(prop);
                if (node.getTypeNode() && !currentSource.isExternal)
                    delayedActions.push(() => handlePropertyOrVariable(prop, node.getTypeNode()));
            }
            else
                parentStack.push(null);
        }
        else
            parentStack.push(null);
    }
}
class VariableDeclarationHandler {
    enter(node) {
        const sym = node.getSymbol();
        const parent = getCurrentParent();
        let from;
        if (sym) {
            if (parent == null || parent instanceof model_1.FunctionOrMethodWithBody) {
                // Don't add local variables to the model
                localSymbols.add(sym);
                from = getLastNonNullParent();
                parentStack.push(null);
            }
            else {
                let v = new model_1.PropertyOrVariable(parent, sym.getName(), node, node.getStartLineNumber(), model_1.ElementType.Variable, node.getInitializer());
                currentSource.addElement(v);
                parentStack.push(v);
                from = v;
            }
            if (node.getTypeNode() && !currentSource.isExternal)
                delayedActions.push(() => handlePropertyOrVariable(from, node.getTypeNode()));
        }
        else
            parentStack.push(null);
    }
}
class GetAccessorHandler {
    enter(node) {
        const sym = node.getSymbol();
        const parent = getCurrentParent();
        if (parent && sym) {
            let getter;
            if (node.isAbstract() || !node.getBody())
                getter = new model_1.FunctionOrMethodWithoutBody(parent, sym.getName(), node, node.getStartLineNumber(), model_1.ElementType.Getter, node.isAbstract());
            else
                getter = new model_1.FunctionOrMethodWithBody(parent, sym.getName(), node, node.getStartLineNumber(), model_1.ElementType.Getter, node.getBody());
            handleParametersAndReturnType(getter, node.getParameters(), node.getTypeParameters(), node.getReturnTypeNode());
            currentSource.addElement(getter);
            if (!currentSource.isExternal) {
                delayedActions.push(() => handleParameterTypes(getter, node.getParameters(), true, node.getReturnTypeNode()));
                if (node.getDecorators().length > 0) {
                    delayedActions.push(() => handleDecorators(getter, node.getDecorators()));
                }
            }
            parentStack.push(getter);
        }
        else
            parentStack.push(null);
    }
}
class SetAccessorHandler {
    enter(node) {
        const sym = node.getSymbol();
        const parent = getCurrentParent();
        if (parent && sym) {
            let setter;
            if (node.getBody())
                setter = new model_1.FunctionOrMethodWithBody(parent, sym.getName(), node, node.getStartLineNumber(), model_1.ElementType.Setter, node.getBody());
            else
                setter = new model_1.FunctionOrMethodWithoutBody(parent, sym.getName(), node, node.getStartLineNumber(), model_1.ElementType.Setter, node.isAbstract());
            handleParametersAndReturnType(setter, node.getParameters(), node.getTypeParameters(), node.getReturnTypeNode());
            currentSource.addElement(setter);
            if (!currentSource.isExternal) {
                delayedActions.push(() => handleParameterTypes(setter, node.getParameters()));
                if (node.getDecorators().length > 0) {
                    delayedActions.push(() => handleDecorators(setter, node.getDecorators()));
                }
            }
            parentStack.push(setter);
        }
        else
            parentStack.push(null);
    }
}
class EnumMemberHandler {
    enter(node) {
        const sym = node.getSymbol();
        const parent = getCurrentParent();
        if (parent && sym) {
            let m = new model_1.PropertyOrVariable(parent, sym.getName(), node, node.getStartLineNumber(), model_1.ElementType.EnumMember, node.getInitializer());
            currentSource.addElement(m);
            parentStack.push(m);
        }
        else
            parentStack.push(null);
    }
}
class NamespaceHandler {
    enter(node) {
        const sym = node.getSymbol();
        const parent = getCurrentParent();
        if (parent && sym) {
            let ns = new model_1.Namespace(parent, sym.getName(), node, node.getStartLineNumber());
            currentSource.addElement(ns);
            parentStack.push(ns);
        }
        else
            parentStack.push(null);
    }
}
function addDependencyFromTypeAlias(ta, type, line) {
    addDependencyToType(ta, type, new Set(), line);
}
class TypeAliasHandler {
    enter(node) {
        const sym = node.getSymbol();
        const parent = getCurrentParent();
        if (parent && sym) {
            const line = node.getStartLineNumber();
            const ta = new model_1.TypeAlias(parent, sym.getName(), node, line);
            let typeParamsAsString = [];
            node.getTypeParameters().forEach(tp => typeParamsAsString.push(tp.getText()));
            ta.typeParameters = typeParamsAsString.join(";");
            currentSource.addElement(ta);
            if (!currentSource.isExternal)
                delayedActions.push(() => addDependencyFromTypeAlias(ta, node.getType(), line));
            parentStack.push(ta);
        }
        else
            parentStack.push(null);
    }
}
class PropertyAssignmentHandler {
    enter(node) {
        const parent = getCurrentParent();
        if (parent instanceof model_1.PropertyOrVariable) {
            const nameNode = node.getNameNode();
            if (isIdentifier(nameNode)) {
                const prop = new model_1.PropertyOrVariable(parent, nameNode.getText(), node, node.getStartLineNumber(), model_1.ElementType.Property, node.getInitializer());
                currentSource.addElement(prop);
                parentStack.push(prop);
            }
            else
                parentStack.push(null);
        }
        else
            parentStack.push(null);
    }
}
class ClassExpressionHandler {
    enter(node) {
        let parent = getCurrentParent();
        if (parent) {
            let cls = new model_1.Class(parent, "", node, node.getStartLineNumber(), false);
            for (const tp of node.getTypeParameters()) {
                const sym = tp.getSymbol();
                if (sym)
                    localSymbols.add(sym);
            }
            if (!currentSource.isExternal)
                delayedActions.push(() => handleInheritance(cls, node.getHeritageClauses()));
            parentStack.push(cls);
        }
        else
            parentStack.push(null);
    }
}
class FunctionExpressionHandler {
    enter(node) {
        let fd;
        let parent = getCurrentParent();
        if (!parent)
            fd = null;
        else
            fd = new model_1.FunctionOrMethodWithBody(parent, "", node, node.getStartLineNumber(), model_1.ElementType.Function, node.getBody());
        if (fd) {
            handleParametersAndReturnType(fd, node.getParameters(), node.getTypeParameters(), node.getReturnTypeNode());
            if (!currentSource.isExternal)
                delayedActions.push(() => handleParameterTypes(fd, node.getParameters(), true, node.getReturnTypeNode()));
        }
        parentStack.push(fd);
    }
}
class ArrowFunctionHandler {
    enter(node) {
        let parent = getCurrentParent();
        if (!parent)
            parent = getLastNonNullParent();
        const fd = new model_1.FunctionOrMethodWithBody(parent, "", node, node.getStartLineNumber(), model_1.ElementType.Function, node.getBody());
        handleParametersAndReturnType(fd, node.getParameters(), node.getTypeParameters(), node.getReturnTypeNode());
        if (!currentSource.isExternal)
            delayedActions.push(() => handleParameterTypes(fd, node.getParameters(), true, node.getReturnTypeNode()));
        parentStack.push(fd);
    }
}
function handleImport(currentSource, to, localName, line) {
    if (to instanceof model_1.Namespace) {
        for (const member of to.children) {
            currentSource.defineImport(localName + "." + member.name, [member]);
        }
    }
    else {
        addDependency(currentSource, to, model_1.DependencyKind.IMPORTS, line);
    }
}
function defineImports(currentSource, localName, name, importSource, line) {
    const importedElements = importSource.findElements(name);
    if (importedElements) {
        currentSource.defineImport(localName, importedElements);
        importedElements.forEach(e => handleImport(currentSource, e, localName, line));
    }
}
function importAllFrom(currentSource, importSource) {
    importSource.elementMap.forEach((v, k) => currentSource.defineImport(k, v));
}
class ImportDeclarationHandler {
    enter(node) {
        if (!currentSource.isExternal) {
            const sourceFile = node.getModuleSpecifierSourceFile();
            if (sourceFile) {
                const sourceFileElement = getSourceFileElement(sourceFile, true);
                if (sourceFileElement) {
                    const src = currentSource;
                    if (sourceFileElement.isExternal)
                        externalSourcesToParse.add(sourceFile);
                    if (node.getNamedImports().length == 0) {
                        delayedActions.push(() => importAllFrom(src, sourceFileElement));
                    }
                    else {
                        node.getNamedImports().forEach(namedImport => {
                            var _a;
                            let localName = (_a = namedImport.getAliasNode()) === null || _a === void 0 ? void 0 : _a.getText();
                            let name = namedImport.getName();
                            if (!localName)
                                localName = name;
                            delayedActions.push(() => defineImports(src, localName, name, sourceFileElement, namedImport.getStartLineNumber()));
                        });
                    }
                }
            }
        }
        parentStack.push(null);
    }
}
function createHandler(node) {
    switch (node.getKind()) {
        case ts_morph_1.SyntaxKind.ImportDeclaration:
            return new ImportDeclarationHandler();
        case ts_morph_1.SyntaxKind.ClassExpression:
            return new ClassExpressionHandler();
        case ts_morph_1.SyntaxKind.FunctionExpression:
            return new FunctionExpressionHandler();
        case ts_morph_1.SyntaxKind.ArrowFunction:
            return new ArrowFunctionHandler();
        case ts_morph_1.SyntaxKind.FunctionDeclaration:
            return new FunctionDeclarationHandler();
        case ts_morph_1.SyntaxKind.Constructor:
            return new ConstructorDeclarationHandler();
        case ts_morph_1.SyntaxKind.MethodDeclaration:
            return new MethodDeclarationHandler();
        case ts_morph_1.SyntaxKind.MethodSignature:
            return new MethodSignatureHandler();
        case ts_morph_1.SyntaxKind.ClassDeclaration:
            return new ClassDeclarationHandler();
        case ts_morph_1.SyntaxKind.InterfaceDeclaration:
            return new InterfaceDeclarationHandler();
        case ts_morph_1.SyntaxKind.EnumDeclaration:
            return new EnumDeclarationHandler();
        case ts_morph_1.SyntaxKind.PropertyDeclaration:
            return new PropertyDeclarationHandler();
        case ts_morph_1.SyntaxKind.PropertySignature:
            return new PropertySignatureHandler();
        case ts_morph_1.SyntaxKind.VariableDeclaration:
            return new VariableDeclarationHandler();
        case ts_morph_1.SyntaxKind.GetAccessor:
            return new GetAccessorHandler();
        case ts_morph_1.SyntaxKind.SetAccessor:
            return new SetAccessorHandler();
        case ts_morph_1.SyntaxKind.EnumMember:
            return new EnumMemberHandler();
        case ts_morph_1.SyntaxKind.ModuleDeclaration:
            return new NamespaceHandler();
        case ts_morph_1.SyntaxKind.TypeAliasDeclaration:
            return new TypeAliasHandler();
        case ts_morph_1.SyntaxKind.PropertyAssignment:
            return new PropertyAssignmentHandler();
        default:
            return null;
    }
}
function traverseNodes(node) {
    const handler = createHandler(node);
    if (handler) {
        handler.enter(node);
        node.forEachChild(n => traverseNodes(n));
        parentStack.pop();
    }
    else if (node.getKind() != ts_morph_1.SyntaxKind.ExportDeclaration && node.getKind() != ts_morph_1.SyntaxKind.ImportDeclaration && node.getKind() != ts_morph_1.SyntaxKind.ArrowFunction) {
        node.forEachChild(n => traverseNodes(n));
    }
}
function isAssignmentToken(token) {
    switch (token.getKind()) {
        case ts_morph_1.SyntaxKind.EqualsToken:
        case ts_morph_1.SyntaxKind.PlusEqualsToken:
        case ts_morph_1.SyntaxKind.MinusEqualsToken:
        case ts_morph_1.SyntaxKind.AsteriskAsteriskEqualsToken:
        case ts_morph_1.SyntaxKind.AsteriskEqualsToken:
        case ts_morph_1.SyntaxKind.SlashEqualsToken:
        case ts_morph_1.SyntaxKind.PercentEqualsToken:
        case ts_morph_1.SyntaxKind.AmpersandEqualsToken:
        case ts_morph_1.SyntaxKind.BarEqualsToken:
        case ts_morph_1.SyntaxKind.CaretEqualsToken:
        case ts_morph_1.SyntaxKind.LessThanLessThanEqualsToken:
        case ts_morph_1.SyntaxKind.GreaterThanGreaterThanGreaterThanEqualsToken:
        case ts_morph_1.SyntaxKind.GreaterThanGreaterThanEqualsToken:
        case ts_morph_1.SyntaxKind.BarBarEqualsToken:
        case ts_morph_1.SyntaxKind.AmpersandAmpersandEqualsToken:
        case ts_morph_1.SyntaxKind.QuestionQuestionEqualsToken:
            return true;
    }
    return false;
}
function addDependency(from, to, kind, line) {
    root.addDependency(from, to, line, kind);
}
const fileMap = new Map();
function processFunctionOrMethod(decl, parent, sym, parameters, isAbstract, type) {
    return findOrCreateModelElement(parent, sym.getName(), decl, type, (parent, name, type, decl, line) => new model_1.FunctionOrMethodWithoutBody(parent, name, decl, line, type, isAbstract), element => handleParameterTypes(element, parameters, false));
}
function findOrCreateModelElement(parent, name, decl, type, factory, afterCreationCallback) {
    const line = decl.getStartLineNumber();
    let found = (0, model_1.lookupNode)(decl);
    if (!found) {
        found = factory(parent, name, type, decl, line);
        if (afterCreationCallback) {
            afterCreationCallback(found);
        }
    }
    return found;
}
function processDeclaration(decl, sourceFileElement) {
    let sym = decl.getSymbol();
    if (!sym) {
        if (decl.getParent())
            return processDeclaration(decl.getParent(), sourceFileElement);
        return sourceFileElement;
    }
    const flags = sym.getFlags();
    let parent;
    if (decl.getParent())
        parent = processDeclaration(decl.getParent(), sourceFileElement);
    else
        parent = sourceFileElement;
    if (!parent)
        return undefined;
    if (decl instanceof ts_morph_1.SourceFile)
        return sourceFileElement;
    if (flags & ts_morph_1.SymbolFlags.Namespace)
        return findOrCreateModelElement(parent, sym.getName(), decl, model_1.ElementType.Namespace, (parent, name, type, decl, line) => new model_1.Namespace(parent, name, decl, line));
    if (flags & ts_morph_1.SymbolFlags.Class) {
        let isAbstract = false;
        let name = sym.getName();
        if (decl instanceof ts_morph_1.ClassDeclaration) {
            const cd = decl;
            isAbstract = cd.isAbstract();
            if (name === "default" && cd.getName())
                name = cd.getName();
        }
        return findOrCreateModelElement(parent, name, decl, model_1.ElementType.Class, (parent, name, type, decl, line) => new model_1.Class(parent, name, decl, line, isAbstract));
    }
    if (flags & ts_morph_1.SymbolFlags.Interface)
        return findOrCreateModelElement(parent, sym.getName(), decl, model_1.ElementType.Interface, (parent, name, type, decl, line) => new model_1.Interface(parent, name, decl, line));
    if (flags & ts_morph_1.SymbolFlags.Function) {
        if (decl instanceof ts_morph_1.FunctionDeclaration) {
            let functionNode = decl;
            return processFunctionOrMethod(decl, parent, sym, functionNode.getParameters(), false, model_1.ElementType.Function);
        }
        return undefined;
    }
    if (flags & ts_morph_1.SymbolFlags.Property)
        return findOrCreateModelElement(parent, sym.getName(), decl, model_1.ElementType.Property, (parent, name, type, decl, line) => new model_1.PropertyOrVariable(parent, name, decl, line, type));
    if (flags & ts_morph_1.SymbolFlags.Method) {
        if (decl instanceof ts_morph_1.MethodDeclaration) {
            const methodNode = decl;
            return processFunctionOrMethod(decl, parent, sym, methodNode.getParameters(), methodNode.isAbstract(), model_1.ElementType.Method);
        }
        if (decl instanceof ts_morph_1.MethodSignature) {
            const methodNode = decl;
            return processFunctionOrMethod(decl, parent, sym, methodNode.getParameters(), false, model_1.ElementType.Method);
        }
        return undefined;
    }
    if (flags & ts_morph_1.SymbolFlags.Variable)
        return findOrCreateModelElement(parent, sym.getName(), decl, model_1.ElementType.Variable, (parent, name, type, decl, line) => new model_1.PropertyOrVariable(parent, name, decl, line, type));
    if (flags & ts_morph_1.SymbolFlags.Enum)
        return findOrCreateModelElement(parent, sym.getName(), decl, model_1.ElementType.Enum, (parent, name, type, decl, line) => new model_1.Enum(parent, name, decl, line));
    if (flags & ts_morph_1.SymbolFlags.EnumMember)
        return findOrCreateModelElement(parent, sym.getName(), decl, model_1.ElementType.EnumMember, (parent, name, type, decl, line) => new model_1.PropertyOrVariable(parent, name, decl, line, type));
    if (flags & ts_morph_1.SymbolFlags.GetAccessor) {
        const getterNode = decl;
        return processFunctionOrMethod(decl, parent, sym, getterNode.getParameters(), false, model_1.ElementType.Getter);
    }
    if (flags & ts_morph_1.SymbolFlags.SetAccessor) {
        const setterNode = decl;
        return processFunctionOrMethod(decl, parent, sym, setterNode.getParameters(), false, model_1.ElementType.Setter);
    }
    if (flags & ts_morph_1.SymbolFlags.Constructor) {
        const constructorDecl = decl;
        return processFunctionOrMethod(decl, parent, sym, constructorDecl.getParameters(), false, model_1.ElementType.Setter);
    }
    if (flags & ts_morph_1.SymbolFlags.TypeAlias)
        return findOrCreateModelElement(parent, sym.getName(), decl, model_1.ElementType.TypeAlias, (parent, name, type, decl, line) => new model_1.TypeAlias(parent, name, decl, line));
    return undefined;
}
function getSourceFileElement(src, isExternal) {
    const path = src.getFilePath();
    if (!path) {
        return undefined;
    }
    let result = fileMap.get(path);
    if (!result) {
        result = new model_1.SourceFileElement(root, path, src, isExternal);
        fileMap.set(path, result);
    }
    return result;
}
function addDependencyToSymbol(from, sym, line, kind) {
    // const originalSym = sym;
    while (sym.getAliasedSymbol())
        sym = sym.getAliasedSymbol();
    if (sym.getFlags() == ts_morph_1.SymbolFlags.ValueModule)
        return;
    let to = (0, model_1.lookupSymbol)(sym);
    if (to) {
        if (to instanceof model_1.Enum && kind == model_1.DependencyKind.READ) {
            return;
        }
        addDependency(from, to, kind, line);
    }
    else if (!localSymbols.has(sym) && (sym.getFlags() & (ts_morph_1.SymbolFlags.Prototype | ts_morph_1.SymbolFlags.BlockScopedVariable /*|SymbolFlags.FunctionScopedVariable*/)) == 0) {
        // External symbol
        let fqn = sym.getFullyQualifiedName();
        if (!fqn.startsWith("__") && sym.getDeclarations().length > 0) {
            const decl = sym.getDeclarations()[0];
            to = (0, model_1.lookupNode)(decl);
            if (!to) {
                let sourceFileElement = getSourceFileElement(decl.getSourceFile(), true);
                if (!sourceFileElement) {
                    return;
                }
                to = processDeclaration(decl, sourceFileElement);
            }
            if (to) {
                addDependency(from, to, kind, line);
            }
        }
    }
}
function addDependencyToNode(from, toNode, kind) {
    const sym = toNode.getSymbol();
    if (sym) {
        addDependencyToSymbol(from, sym, toNode.getStartLineNumber(), kind);
    }
}
function addDependencyToType(from, type, visitedTypes, line, dependencyKind = model_1.DependencyKind.USES) {
    if (visitedTypes.has(type))
        return;
    visitedTypes.add(type);
    const targetType = type.getTargetType();
    if (targetType && targetType !== type) {
        addDependencyToType(from, targetType, visitedTypes, line, dependencyKind);
        for (let ta of type.getTypeArguments())
            addDependencyToType(from, ta, visitedTypes, line, dependencyKind);
        return;
    }
    if (type.isClassOrInterface() || type.isEnum()) {
        const sym = type.getSymbol();
        if (sym) {
            addDependencyToSymbol(from, sym, line, dependencyKind);
        }
    }
    else if (type.isUnion()) {
        type.getUnionTypes().forEach(ut => addDependencyToType(from, ut, visitedTypes, line, dependencyKind));
    }
    else if (type.isIntersection()) {
        type.getIntersectionTypes().forEach(it => addDependencyToType(from, it, visitedTypes, line, dependencyKind));
    }
    else if (type.isTuple()) {
        type.getTupleElements().forEach(te => addDependencyToType(from, te, visitedTypes, line, dependencyKind));
    }
    else if (type.isArray()) {
        addDependencyToType(from, type.getArrayElementType(), visitedTypes, line, dependencyKind);
    }
    else if (type.isEnumLiteral()) {
        addDependencyToType(from, type.getBaseTypeOfLiteralType(), visitedTypes, line, dependencyKind);
    }
}
function isIdentifier(node) {
    return node.getKind() == ts_morph_1.SyntaxKind.Identifier || node.getKind() == ts_morph_1.SyntaxKind.PrivateIdentifier;
}
function handleBinaryExpression(node, forbiddenNodes, from) {
    const isAssignmentOperator = isAssignmentToken(node.getOperatorToken());
    const leftNode = node.getChildAtIndex(0);
    const rightNode = node.getChildAtIndex(2);
    if (isAssignmentOperator && isIdentifier(leftNode)) {
        addDependencyToNode(from, leftNode, model_1.DependencyKind.WRITE);
    }
    else {
        detectDependencies(leftNode, forbiddenNodes, from, isAssignmentOperator ? model_1.DependencyKind.WRITE : model_1.DependencyKind.READ);
    }
    detectDependencies(rightNode, forbiddenNodes, from, model_1.DependencyKind.READ);
}
function handlePropertyAccess(node, forbiddenNodes, from, dependencyKind) {
    const leftNode = node.getChildAtIndex(0);
    const rightNode = node.getChildAtIndex(2);
    if (isIdentifier(leftNode)) {
        addDependencyToNode(from, leftNode, model_1.DependencyKind.READ);
    }
    else {
        detectDependencies(leftNode, forbiddenNodes, from, model_1.DependencyKind.READ);
    }
    if (isIdentifier(rightNode)) {
        addDependencyToNode(from, rightNode, dependencyKind);
    }
}
function handleCallExpression(node, forbiddenNodes, from) {
    const expr = node.getExpression();
    if (isIdentifier(expr)) {
        addDependencyToNode(from, expr, model_1.DependencyKind.CALL);
    }
    else {
        detectDependencies(node.getExpression(), forbiddenNodes, from, model_1.DependencyKind.CALL);
    }
    node.getArguments().forEach(arg => detectDependencies(arg, forbiddenNodes, from, model_1.DependencyKind.READ));
}
function handlePostfixUnaryExpression(node, forbiddenNodes, from, dependencyKind) {
    const operand = node.getOperand();
    if (isIdentifier(operand)) {
        addDependencyToNode(from, operand, model_1.DependencyKind.WRITE);
    }
    else {
        detectDependencies(operand, forbiddenNodes, from, dependencyKind);
    }
}
function handlePrefixUnaryExpression(node, forbiddenNodes, from) {
    const operator = node.getOperatorToken();
    const operand = node.getOperand();
    const isIncOrDec = operator == ts_morph_1.SyntaxKind.PlusPlusToken || operator == ts_morph_1.SyntaxKind.MinusMinusToken;
    if (isIncOrDec && isIdentifier(operand)) {
        addDependencyToNode(from, operand, model_1.DependencyKind.WRITE);
    }
    else {
        detectDependencies(operand, forbiddenNodes, from, isIncOrDec ? model_1.DependencyKind.WRITE : model_1.DependencyKind.READ);
    }
}
function handleNewExpression(node, forbiddenNodes, from) {
    const expr = node.getExpression();
    if (isIdentifier(expr)) {
        addDependencyToNode(from, expr, model_1.DependencyKind.NEW);
    }
    else {
        // Likely PropertyAccessExpression
        detectDependencies(expr, forbiddenNodes, from, model_1.DependencyKind.NEW);
    }
    node.getArguments().forEach(arg => detectDependencies(arg, forbiddenNodes, from, model_1.DependencyKind.READ));
}
function handleTypeReference(node, forbiddenNodes, from) {
    addDependencyToType(from, node.getTypeName().getType(), new Set(), node.getStartLineNumber());
}
function detectDependencies(node, forbiddenNodes, from, dependencyKind) {
    if (forbiddenNodes.has(node))
        return;
    switch (node.getKind()) {
        case ts_morph_1.SyntaxKind.BinaryExpression:
            handleBinaryExpression(node, forbiddenNodes, from);
            break;
        case ts_morph_1.SyntaxKind.PropertyAccessExpression:
            handlePropertyAccess(node, forbiddenNodes, from, dependencyKind);
            break;
        case ts_morph_1.SyntaxKind.CallExpression:
            handleCallExpression(node, forbiddenNodes, from);
            break;
        case ts_morph_1.SyntaxKind.PostfixUnaryExpression:
            handlePostfixUnaryExpression(node, forbiddenNodes, from, model_1.DependencyKind.READ);
            break;
        case ts_morph_1.SyntaxKind.PrefixUnaryExpression:
            handlePrefixUnaryExpression(node, forbiddenNodes, from);
            break;
        case ts_morph_1.SyntaxKind.NewExpression:
            handleNewExpression(node, forbiddenNodes, from);
            break;
        case ts_morph_1.SyntaxKind.Identifier:
        case ts_morph_1.SyntaxKind.PrivateIdentifier:
            addDependencyToNode(from, node, model_1.DependencyKind.READ);
            break;
        case ts_morph_1.SyntaxKind.TypeReference:
            handleTypeReference(node, forbiddenNodes, from);
            break;
        case ts_morph_1.SyntaxKind.ExportDeclaration:
        case ts_morph_1.SyntaxKind.ImportDeclaration:
            break;
        case ts_morph_1.SyntaxKind.ModuleDeclaration:
            const md = node;
            if (md.getBody())
                detectDependencies(md.getBody(), forbiddenNodes, from, dependencyKind);
            break;
        case ts_morph_1.SyntaxKind.TypeParameter:
            const decl = node;
            if (decl.getConstraint()) {
                detectDependencies(decl.getConstraint(), forbiddenNodes, from, dependencyKind);
            }
            break;
        case ts_morph_1.SyntaxKind.PropertyAssignment:
            const pa = node;
            if (pa.getInitializer())
                detectDependencies(pa.getInitializer(), forbiddenNodes, from, dependencyKind);
            break;
        case ts_morph_1.SyntaxKind.FunctionDeclaration:
            const fd = node;
            if (fd.getBody())
                detectDependencies(fd.getBody(), forbiddenNodes, from, dependencyKind);
            break;
        case ts_morph_1.SyntaxKind.ArrowFunction:
            const af = node;
            if (af.getBody())
                detectDependencies(af.getBody(), forbiddenNodes, from, dependencyKind);
            break;
        default:
            node.forEachChild(cn => detectDependencies(cn, forbiddenNodes, from, dependencyKind));
            break;
    }
}
function visitSourceFiles(languageService, tsConfigPath, sourceFiles) {
    root = new model_1.Root(tsConfigPath);
    // First let's make sure all internal SourceFileElement's are actually created
    for (let sourceFile of sourceFiles) {
        getSourceFileElement(sourceFile, false);
    }
    // Now let's build a model
    for (let sourceFile of sourceFiles) {
        const sf = getSourceFileElement(sourceFile, false);
        if (!sf) {
            continue;
        }
        let hasStatements = false;
        parentStack.push(sf);
        currentSource = sf;
        sourceFile.forEachChild(n => {
            traverseNodes(n);
            if (n.getKind() >= ts_morph_1.SyntaxKind.FirstStatement && n.getKind() <= ts_morph_1.SyntaxKind.LastStatement && n.getKind() != ts_morph_1.SyntaxKind.VariableStatement)
                hasStatements = true;
        });
        sf.hasBeenParsed = true;
        if (hasStatements) {
            const mc = new metric_collector_1.MetricCollector(sourceFile);
            mc.collectMetrics();
            sf.numberOfStatements = mc.numberOfStatements;
            sf.cyclomaticComplexity = mc.cyclomaticComplexity;
            sf.modifiedCyclomaticComplexity = mc.modifiedCyclomaticComplexity;
            sf.logicalOperations = mc.logicalOperations;
            sf.maxIndent = mc.maxIndent;
        }
        parentStack.pop();
    }
    for (let sourceFile of externalSourcesToParse) {
        const sf = getSourceFileElement(sourceFile, false);
        if (sf) {
            parentStack.push(sf);
            currentSource = sf;
            sourceFile.forEachChild(n => {
                traverseNodes(n);
            });
            sf.hasBeenParsed = true;
            parentStack.pop();
        }
    }
    delayedActions.forEach(action => action());
    delayedActions = [];
    let elements = (0, model_1.getAllElements)();
    const upperIndex = elements.length;
    for (let i = 0; i < upperIndex; i++) {
        let element = elements[i];
        if (element.isExternal) {
            continue;
        }
        if (element instanceof model_1.FunctionOrMethodWithBody) {
            const childrenNodes = new Set();
            for (let child of element.children) {
                if (!(child instanceof model_1.PropertyOrVariable)) {
                    childrenNodes.add(child.node);
                }
            }
            if (element.bodyNode) {
                detectDependencies(element.bodyNode, childrenNodes, element, model_1.DependencyKind.READ);
            }
        }
        else if (element instanceof model_1.PropertyOrVariable) {
            if (element.initializer)
                detectDependencies(element.initializer, emptySet, element, model_1.DependencyKind.READ);
        }
        else if (element instanceof model_1.SourceFileElement) {
            const childrenNodes = new Set();
            element.children.forEach(c => childrenNodes.add(c.node));
            detectDependencies(element.node, childrenNodes, element, model_1.DependencyKind.READ);
        }
    }
    return root;
}
//# sourceMappingURL=visitor.js.map