"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MetricCollector = void 0;
const ts_morph_1 = require("ts-morph");
class MetricCollector {
    constructor(...bodyNodes) {
        this._numberOfStatements = 0;
        this._cyclomaticComplexity = 1;
        this._modifiedCyclomaticComplexity = 1;
        this._logicalOperations = 0;
        this._maxIndent = 0;
        this._currentIndent = 0;
        this._nodes = [];
        this._nodes = bodyNodes;
    }
    collectMetrics() {
        for (let node of this._nodes) {
            this.traverse(node);
        }
    }
    get numberOfStatements() {
        return this._numberOfStatements;
    }
    get cyclomaticComplexity() {
        return this._cyclomaticComplexity;
    }
    get modifiedCyclomaticComplexity() {
        return this._modifiedCyclomaticComplexity;
    }
    get maxIndent() {
        return this._maxIndent;
    }
    get logicalOperations() {
        return this._logicalOperations;
    }
    indent() {
        this._currentIndent++;
        this._maxIndent = Math.max(this._maxIndent, this._currentIndent);
    }
    dedent() {
        this._currentIndent--;
    }
    traverse(node) {
        switch (node.getKind()) {
            // Skip nested declarations
            case ts_morph_1.SyntaxKind.ClassDeclaration:
            case ts_morph_1.SyntaxKind.ClassExpression:
            case ts_morph_1.SyntaxKind.FunctionExpression:
            case ts_morph_1.SyntaxKind.InterfaceDeclaration:
            case ts_morph_1.SyntaxKind.FunctionDeclaration:
            case ts_morph_1.SyntaxKind.EnumDeclaration:
            case ts_morph_1.SyntaxKind.ArrowFunction:
                return;
        }
        if (node.getKind() >= ts_morph_1.SyntaxKind.FirstStatement && node.getKind() <= ts_morph_1.SyntaxKind.LastStatement) {
            this._numberOfStatements++;
        }
        switch (node.getKind()) {
            case ts_morph_1.SyntaxKind.ForStatement:
            case ts_morph_1.SyntaxKind.ForOfStatement:
            case ts_morph_1.SyntaxKind.ForInStatement:
            case ts_morph_1.SyntaxKind.IfStatement:
            case ts_morph_1.SyntaxKind.WhileStatement:
            case ts_morph_1.SyntaxKind.DoStatement:
                this._cyclomaticComplexity++;
                this._modifiedCyclomaticComplexity++;
                break;
            case ts_morph_1.SyntaxKind.SwitchStatement:
                this._modifiedCyclomaticComplexity++;
                break;
            case ts_morph_1.SyntaxKind.CaseBlock:
                this._cyclomaticComplexity++;
                break;
            case ts_morph_1.SyntaxKind.BinaryExpression:
                const be = node;
                const ot = be.getOperatorToken().getKind();
                if (ot == ts_morph_1.SyntaxKind.AmpersandAmpersandToken || ot == ts_morph_1.SyntaxKind.BarBarToken)
                    this._logicalOperations++;
                break;
        }
        switch (node.getKind()) {
            case ts_morph_1.SyntaxKind.ForStatement:
                const fs = node;
                if (fs.getCondition())
                    this.traverse(fs.getCondition());
                this.indent();
                this.traverse(fs.getStatement());
                this.dedent();
                break;
            case ts_morph_1.SyntaxKind.ForOfStatement:
                const fo = node;
                this.indent();
                this.traverse(fo.getStatement());
                this.dedent();
                break;
            case ts_morph_1.SyntaxKind.ForInStatement:
                const fi = node;
                this.indent();
                this.traverse(fi.getStatement());
                this.dedent();
                break;
            case ts_morph_1.SyntaxKind.IfStatement:
                const ifstmt = node;
                this.traverse(ifstmt.getExpression());
                this.indent();
                this.traverse(ifstmt.getThenStatement());
                this.dedent();
                const es = ifstmt.getElseStatement();
                if (es) {
                    if (es.getKind() == ts_morph_1.SyntaxKind.IfStatement) {
                        // Special treatment for "else if"
                        this.traverse(es);
                    }
                    else {
                        this.indent();
                        this.traverse(es);
                        this.dedent();
                    }
                }
                break;
            case ts_morph_1.SyntaxKind.WhileStatement:
                const ws = node;
                this.traverse(ws.getExpression());
                this.indent();
                this.traverse(ws.getStatement());
                this.dedent();
                break;
            case ts_morph_1.SyntaxKind.DoStatement:
                const ds = node;
                this.traverse(ds.getExpression());
                this.indent();
                this.traverse(ds.getStatement());
                this.dedent();
                break;
            case ts_morph_1.SyntaxKind.SwitchStatement:
                const st = node;
                this.indent();
                st.getClauses().forEach(clause => this.traverse(clause));
                this.dedent();
                break;
            case ts_morph_1.SyntaxKind.TryStatement:
                const ts = node;
                this.indent();
                this.traverse(ts.getTryBlock());
                if (ts.getCatchClause())
                    this.traverse(ts.getCatchClause());
                if (ts.getFinallyBlock())
                    this.traverse((ts.getFinallyBlock()));
                this.dedent();
                break;
            case ts_morph_1.SyntaxKind.WithStatement:
                const wis = node;
                this.indent();
                this.traverse(wis.getStatement());
                this.dedent();
                break;
            default:
                node.forEachChild(cn => this.traverse(cn));
                break;
        }
    }
}
exports.MetricCollector = MetricCollector;
//# sourceMappingURL=metric_collector.js.map