/*
 * Decompiled with CFR 0.152.
 */
package com.hello2morrow.sonargraph.foundation.windows;

import com.hello2morrow.sonargraph.foundation.file.FileUtility;
import com.hello2morrow.sonargraph.foundation.file.TDirectoryFileFilter;
import com.hello2morrow.sonargraph.foundation.utilities.Platform;
import com.hello2morrow.sonargraph.foundation.utilities.Version;
import com.hello2morrow.sonargraph.foundation.windows.WinRegistry;
import de.schlichtherle.truezip.file.TFile;
import java.io.File;
import java.io.FileFilter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VisualStudioInstallationsDetector {
    public static final String DEFAULT_MSBUILD_INSTALLATION_PATH = "C:/Program Files (x86)/MSBuild";
    private static final Logger LOGGER = LoggerFactory.getLogger(VisualStudioInstallationsDetector.class);
    private static final String COMMON7_TOOLS_VS_DEV_CMD_BAT = "Common7/Tools/VsDevCmd.bat";
    public static final String REG_KEY_64 = "SOFTWARE\\Wow6432Node\\Microsoft\\VisualStudio\\SxS\\VC7";
    public static final String REG_KEY_64_2017 = "SOFTWARE\\Wow6432Node\\Microsoft\\VisualStudio\\SxS\\VS7";
    public static final String REG_KEY_32 = "SOFTWARE\\Microsoft\\VisualStudio\\SxS\\VC7";
    public static final String VS_COMMAND_PROMPT_SCRIPT = "vcvarsall.bat";
    public static final String VS_DEVELOPER_COMMAND_PROMPT_SCRIPT = "VsDevCmd.bat";

    private VisualStudioInstallationsDetector() {
    }

    public static List<Location> getVisualStudioLocations() {
        Map<String, String> vs2017LocationsMap;
        if (!Platform.isWindows()) {
            throw new RuntimeException("Only applicable for Windows machines");
        }
        boolean is64Bit = Platform.isOperatingSystem64Bit();
        String key = is64Bit ? REG_KEY_64 : REG_KEY_32;
        Map<String, String> standardLocationsMap = WinRegistry.readStringValues(-2147483646L, key);
        if (is64Bit && (vs2017LocationsMap = WinRegistry.readStringValues(-2147483646L, REG_KEY_64_2017)) != null) {
            if (standardLocationsMap == null) {
                standardLocationsMap = new HashMap<String, String>();
            }
            for (Map.Entry<String, String> entry2 : vs2017LocationsMap.entrySet()) {
                String k = entry2.getKey();
                if (k == null || k.length() <= 0) continue;
                standardLocationsMap.put(k, entry2.getValue());
            }
        }
        if (standardLocationsMap == null) {
            LOGGER.error("Failed to analyze Windows Registry to detect Visual Studio Compiler installations");
            return Collections.emptyList();
        }
        return standardLocationsMap.entrySet().stream().map(entry -> new Location((String)entry.getKey(), new TFile((String)entry.getValue()))).collect(Collectors.toList());
    }

    public static Optional<Location> getLatestVisualStudioLocation() {
        return VisualStudioInstallationsDetector.getLatestLocation(VisualStudioInstallationsDetector.getVisualStudioLocations());
    }

    public static boolean is2017VersionOrNewer(String version) {
        assert (version != null && version.length() > 0) : "Parameter 'version' of method 'is2017VersionOrNewer' must not be empty";
        try {
            Double number = Double.parseDouble(version);
            return number >= 15.0;
        }
        catch (NumberFormatException ex) {
            LOGGER.info("Version is not a number: " + version);
            return false;
        }
    }

    public static Optional<Location> getLatestLocation(List<Location> locations) {
        assert (locations != null) : "Parameter 'locations' of method 'getLatestLocation' must not be null";
        double highestVersion = 0.0;
        Location latest = null;
        for (Location next : locations) {
            double version;
            String versionString = next.getVersion();
            if (!VisualStudioInstallationsDetector.isConvertibleVersion(versionString) || !((version = Double.valueOf(versionString).doubleValue()) > highestVersion)) continue;
            highestVersion = version;
            latest = next;
        }
        return Optional.ofNullable(latest);
    }

    public static boolean isConvertibleVersion(String version) {
        boolean patternMatch;
        boolean bl = patternMatch = version != null && version.matches("[\\d\\.]+");
        if (!patternMatch) {
            return false;
        }
        try {
            Double.parseDouble(version);
            return true;
        }
        catch (NumberFormatException ex) {
            LOGGER.info("Not a convertible version string: " + version);
            return false;
        }
    }

    public static List<Location> getVisualStudioCommandPromptsByRegistry() {
        return VisualStudioInstallationsDetector.getVisualStudioLocations().stream().map(l -> new Location(l.getVersion(), new TFile((File)l.getFile(), COMMON7_TOOLS_VS_DEV_CMD_BAT))).filter(l -> l.getFile().exists()).collect(Collectors.toList());
    }

    public static List<Location> getVisualStudioCommandPromptsByStandardLocation() {
        ArrayList<Location> locations = new ArrayList<Location>();
        TFile rootDir = new TFile("C:/Program Files (x86)");
        if (!(rootDir.exists() && rootDir.isDirectory() && rootDir.canRead())) {
            LOGGER.debug("Failed to determine Visual Studio Developer Prompts. Standard location '" + rootDir.getAbsolutePath() + "' does not exist or is not readable.");
            return locations;
        }
        String regex = "Microsoft Visual Studio( [\\d.]+)?.*";
        Pattern pattern = Pattern.compile("Microsoft Visual Studio( [\\d.]+)?.*");
        TFile[] tFileArray = FileUtility.listFilesInDir(rootDir, "Microsoft Visual Studio( [\\d.]+)?.*");
        int n = tFileArray.length;
        int n2 = 0;
        while (n2 < n) {
            TFile candidate = tFileArray[n2];
            if (!candidate.isDirectory() || !candidate.canRead()) {
                LOGGER.debug("Directory '" + candidate.getAbsolutePath() + "' is not accessible.");
            } else {
                String folderName = candidate.getName();
                Matcher matcher = pattern.matcher(folderName);
                if (matcher.find()) {
                    String versionString = matcher.group(1);
                    if (versionString != null) {
                        TFile batchFile2 = new TFile((File)candidate, COMMON7_TOOLS_VS_DEV_CMD_BAT);
                        if (batchFile2.exists()) {
                            locations.add(new Location(versionString, batchFile2));
                        }
                    } else {
                        TFile[] tFileArray2 = FileUtility.listFilesInDir(candidate, "\\d+");
                        int n3 = tFileArray2.length;
                        int n4 = 0;
                        while (n4 < n3) {
                            TFile nestedCandidate = tFileArray2[n4];
                            String version = nestedCandidate.getName();
                            TFile[] tFileArray3 = nestedCandidate.listFiles();
                            int n5 = tFileArray3.length;
                            int n6 = 0;
                            while (n6 < n5) {
                                TFile batchFile;
                                TFile common7;
                                TFile edition = tFileArray3[n6];
                                if (edition.isDirectory() && (common7 = new TFile((File)edition, "Common7")).isDirectory() && (batchFile = new TFile((File)edition, COMMON7_TOOLS_VS_DEV_CMD_BAT)).exists()) {
                                    locations.add(new Location(version, batchFile));
                                    break;
                                }
                                ++n6;
                            }
                            ++n4;
                        }
                    }
                }
            }
            ++n2;
        }
        return locations;
    }

    public static Optional<Location> guessLatestVisualStudioCommandPrompt() {
        Optional<Location> location;
        List<Location> visualStudioCommandPrompts;
        Optional<Location> location2;
        List<Location> standardLocations = VisualStudioInstallationsDetector.getVisualStudioCommandPromptsByStandardLocation();
        ArrayList<Location> latest = new ArrayList<Location>();
        if (standardLocations.size() > 0 && (location2 = VisualStudioInstallationsDetector.getLatestLocation(standardLocations)).isPresent()) {
            latest.add(location2.get());
        }
        if ((visualStudioCommandPrompts = VisualStudioInstallationsDetector.getVisualStudioCommandPromptsByRegistry()).size() > 0 && (location = VisualStudioInstallationsDetector.getLatestLocation(visualStudioCommandPrompts)).isPresent()) {
            latest.add(location.get());
        }
        if (latest.isEmpty()) {
            return Optional.empty();
        }
        return VisualStudioInstallationsDetector.getLatestLocation(latest);
    }

    public static Optional<Location> getLatestMsBuild() {
        List<Location> locations = VisualStudioInstallationsDetector.getMsBuildLocations();
        if (locations.isEmpty()) {
            return Optional.empty();
        }
        return VisualStudioInstallationsDetector.getLatestLocation(locations);
    }

    public static List<Location> getMsBuildLocations() {
        TFile instDir;
        Optional<Location> latestMsBuild;
        TFile baseDir = new TFile(DEFAULT_MSBUILD_INSTALLATION_PATH);
        if (!baseDir.exists() || !baseDir.isDirectory()) {
            return Collections.emptyList();
        }
        ArrayList<Location> locations = new ArrayList<Location>();
        Pattern pattern = Pattern.compile("(\\d+\\.\\d+).*");
        TFile[] tFileArray = FileUtility.listFilesInDir(baseDir, "\\d+\\.\\d+.*");
        int n = tFileArray.length;
        int n2 = 0;
        while (n2 < n) {
            TFile next = tFileArray[n2];
            String folderName = next.getName();
            Matcher matcher = pattern.matcher(folderName);
            if (matcher.find()) {
                String version = matcher.group(0);
                TFile executable = new TFile((File)next, "Bin/MSBuild.exe");
                if (executable.exists()) {
                    locations.add(new Location(version, executable));
                }
            }
            ++n2;
        }
        Optional<Location> latestLocation = VisualStudioInstallationsDetector.getLatestVisualStudioLocation();
        if (latestLocation.isPresent() && VisualStudioInstallationsDetector.is2017VersionOrNewer(latestLocation.get().getVersion()) && (latestMsBuild = VisualStudioInstallationsDetector.getLatestMsBuildFor2017OrNewer(instDir = latestLocation.get().getFile())).isPresent()) {
            locations.add(latestMsBuild.get());
        }
        return locations;
    }

    private static Optional<Location> getLatestMsBuildFor2017OrNewer(TFile instDir) {
        assert (instDir != null) : "Parameter 'instDir' of method 'getLatestMsBuildFor2017OrNewer' must not be null";
        TFile msBuildBaseDir = new TFile((File)instDir, "MSBuild");
        if (!msBuildBaseDir.exists() || !msBuildBaseDir.isDirectory()) {
            return Optional.empty();
        }
        ArrayList<Location> msBuildLocations = new ArrayList<Location>();
        TFile[] tFileArray = msBuildBaseDir.listFiles((FileFilter)new TDirectoryFileFilter());
        int n = tFileArray.length;
        int n2 = 0;
        while (n2 < n) {
            TFile next = tFileArray[n2];
            if (VisualStudioInstallationsDetector.isConvertibleVersion(next.getName())) {
                String version = next.getName();
                TFile msBuildExe = new TFile((File)next, "Bin/MSBuild.exe");
                if (msBuildExe.exists()) {
                    msBuildLocations.add(new Location(version, msBuildExe));
                }
            }
            ++n2;
        }
        return VisualStudioInstallationsDetector.getLatestLocation(msBuildLocations);
    }

    public static List<Version> getToolsVersions() {
        if (!Platform.isWindows()) {
            return Collections.emptyList();
        }
        LinkedHashSet<Version> versions = new LinkedHashSet<Version>();
        String key = "Software\\Microsoft\\MSBuild\\ToolsVersions";
        List<String> subkeys = WinRegistry.readStringSubKeys(-2147483646, "Software\\Microsoft\\MSBuild\\ToolsVersions");
        if (subkeys != null) {
            for (String next : subkeys) {
                LOGGER.debug("Found registry key for version: " + next);
                Version version = Version.create(next);
                if (version == null) continue;
                versions.add(version);
            }
        }
        List versionsFromDisk = VisualStudioInstallationsDetector.getMsBuildLocations().stream().map(l -> Version.create(l.getVersion())).collect(Collectors.toList());
        versions.addAll(versionsFromDisk);
        ArrayList<Version> result = new ArrayList<Version>(versions);
        result.sort(new Version.VersionComparator());
        return result;
    }

    public static class Location {
        private final String m_version;
        private final TFile m_location;

        public Location(String version, TFile location) {
            assert (version != null) : "Parameter 'version' of method 'Location' must not be null";
            assert (location != null) : "Parameter 'location' of method 'Location' must not be null";
            this.m_version = version;
            this.m_location = location.getNormalizedAbsoluteFile();
        }

        public TFile getFile() {
            return this.m_location;
        }

        public String getVersion() {
            return this.m_version;
        }

        public String toString() {
            return "version=" + this.getVersion() + ", file=" + this.getFile().getAbsolutePath();
        }

        public boolean isValidLocation() {
            TFile file = new TFile((File)this.m_location, "VC");
            return file.isDirectory();
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * this.m_location.hashCode() + this.m_version.hashCode();
            return result;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            Location other = (Location)obj;
            return this.m_location.equals((Object)other.m_location) && this.m_version.equals(other.m_version);
        }
    }
}

